<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Auth extends CI_Controller
{
	private $db_connected = false;

	public function __construct()
	{
		parent::__construct();
		
		// Load basic helpers first
		$this->load->helper('url');
		
		// Try to connect to database
		try {
			$this->load->database();
			$this->db_connected = true;
		} catch (Exception $e) {
			$this->db_connected = false;
		}
		
		// Load other libraries only if database is connected
		if ($this->db_connected) {
			$this->load->library('form_validation');
			$this->load->library('session');
			$this->load->helper('security');
			
			// Load model if exists
			if (file_exists(APPPATH . 'models/M_data.php')) {
				$this->load->model('m_data');
			}
		}
	}

	public function index()
	{
		// Check if database is connected
		if (!$this->db_connected) {
			$this->show_database_error();
			return;
		}

		// Set validation rules
		$this->form_validation->set_rules('username', 'Username', 'required|trim', ['required' => 'Username harus di isi!']);
		$this->form_validation->set_rules('password', 'Password', 'required|trim', ['required' => 'Password harus di isi!']);

		if ($this->form_validation->run() == TRUE) {
			
			// Get input data with XSS filtering
			$username = $this->security->xss_clean($this->input->post('username', TRUE));
			$password = $this->security->xss_clean($this->input->post('password', TRUE));

			try {
				// Check in admin table
				$admin = $this->db->get_where('tb_admin', ['username' => $username])->row_array();
				
				// Check in guru table
				$guru = $this->db->get_where('tb_guru', ['username' => $username])->row_array();
				
				// Check in siswa table
				$siswa = $this->db->get_where('tb_siswa', ['username' => $username])->row_array();

				// Verify admin login
				if ($admin && $this->verify_password($password, $admin['password'])) {
					$data_session = array(
						'id'       => $admin['id'],
						'nama'     => $admin['nama_user'],
						'username' => $admin['username'],
						'status'   => 'admin_login',
						'logged_in' => TRUE
					);
					$this->session->set_userdata($data_session);
					$this->session->set_flashdata('message', '<div class="alert alert-success alert-message text-center"><b>Login Berhasil!<br></b> Anda akan diarahkan ke halaman admin.</div>');
					redirect(base_url('home'));

				} 
				// Verify guru login
				else if ($guru && $this->verify_password($password, $guru['password'])) {
					$data_session = array(
						'id'       => $guru['id_guru'],
						'id_guru'  => $guru['id_guru'],
						'nama'     => $guru['nama_guru'],
						'username' => $guru['username'],
						'status'   => 'guru_login',
						'logged_in' => TRUE
					);
					$this->session->set_userdata($data_session);
					$this->session->set_flashdata('message', '<div class="alert alert-success alert-message text-center"><b>Login Berhasil!<br></b> Anda akan diarahkan ke halaman guru.</div>');
					redirect(base_url('home'));

				} 
				// Verify siswa login
				else if ($siswa && $this->verify_password($password, $siswa['password'])) {
					$data_session = array(
						'id'       => $siswa['id_siswa'],
						'nis'      => $siswa['nis'],
						'nama'     => $siswa['nama_siswa'],
						'username' => $siswa['username'],
						'status'   => 'siswa_login',
						'logged_in' => TRUE
					);
					$this->session->set_userdata($data_session);
					$this->session->set_flashdata('message', '<div class="alert alert-success alert-message text-center"><b>Login Berhasil!<br></b> Anda akan diarahkan ke halaman siswa.</div>');
					redirect(base_url('home_siswa'));

				} else {
					$this->session->set_flashdata('message', '<div class="alert alert-danger alert-message text-center"><b>Login Gagal!<br></b> Username atau Password tidak cocok.</div>');
					redirect(base_url('auth'));
				}
				
			} catch (Exception $e) {
				$this->session->set_flashdata('message', '<div class="alert alert-danger alert-message text-center"><b>Error!</b> Terjadi kesalahan sistem.</div>');
				redirect(base_url('auth'));
			}

		} else {
			// Load login view
			$data['title'] = 'Login';
			$this->load->view('v_login', $data);
		}
	}

	/**
	 * Show database connection error page
	 */
	private function show_database_error()
	{
		$data['title'] = 'Database Error';
		$data['message'] = 'Tidak dapat terhubung ke database. Silakan hubungi administrator.';
		
		// Create simple error view if v_login doesn't exist
		if (!file_exists(APPPATH . 'views/v_login.php')) {
			echo '<!DOCTYPE html>
<html>
<head>
    <title>Database Error</title>
    <style>
        body { font-family: Arial, sans-serif; text-align: center; margin-top: 50px; }
        .error { color: red; padding: 20px; border: 1px solid red; display: inline-block; }
    </style>
</head>
<body>
    <div class="error">
        <h2>Database Connection Error</h2>
        <p>Tidak dapat terhubung ke database. Silakan periksa konfigurasi database Anda.</p>
    </div>
</body>
</html>';
		} else {
			$this->load->view('v_login', $data);
		}
	}

	/**
	 * Verify password - supports both plain text and hashed passwords
	 * For security, consider migrating to password_hash() and password_verify()
	 */
	private function verify_password($input_password, $stored_password)
	{
		// If stored password is hashed (starts with $2y$ for bcrypt)
		if (substr($stored_password, 0, 4) === '$2y$' || substr($stored_password, 0, 4) === '$2a$') {
			return password_verify($input_password, $stored_password);
		}
		// For legacy plain text passwords (not recommended for production)
		else {
			return $input_password === $stored_password;
		}
	}

	/**
	 * Logout function
	 */
	public function logout()
	{
		$this->session->sess_destroy();
		$this->session->set_flashdata('message', '<div class="alert alert-info alert-message text-center"><b>Logout Berhasil!</b></div>');
		redirect(base_url('auth'));
	}
}