<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Auth extends CI_Controller
{
    private $db_connected = false;
    private $use_database = false;

    public function __construct()
    {
        // Start output buffering to catch any unwanted output
        if (!ob_get_level()) {
            ob_start();
        }

        parent::__construct();
        
        // Load essential libraries and helpers
        $this->load->helper('url');
        $this->load->library('form_validation');
        $this->load->helper('security');
        
        // Initialize session handling
        $this->initialize_session();
        
        // Try to load database and model (silently handle errors)
        $this->try_initialize_database();
    }

    private function initialize_session()
    {
        // Load CI session library with error handling
        try {
            // Suppress output during session loading
            ob_start();
            $this->load->library('session');
            ob_end_clean();
        } catch (Exception $e) {
            // Clean buffer if there was output
            if (ob_get_level()) {
                ob_end_clean();
            }
            
            // If CI session fails, start native PHP session as fallback
            if (session_status() == PHP_SESSION_NONE) {
                ini_set('session.cookie_httponly', 1);
                ini_set('session.use_strict_mode', 1);
                ini_set('session.cookie_secure', 0);
                @session_start();
            }
            
            // Create mock session object for compatibility
            $this->create_mock_session();
        }
    }

    private function create_mock_session()
    {
        $this->session = new stdClass();
        $this->session->set_userdata = function($data) {
            if (is_array($data)) {
                foreach ($data as $key => $value) {
                    $_SESSION[$key] = $value;
                }
            }
        };
        $this->session->set_flashdata = function($key, $value) {
            if (!isset($_SESSION['__flash_data'])) {
                $_SESSION['__flash_data'] = [];
            }
            $_SESSION['__flash_data'][$key] = $value;
        };
        $this->session->flashdata = function($key) {
            $value = isset($_SESSION['__flash_data'][$key]) ? $_SESSION['__flash_data'][$key] : null;
            if (isset($_SESSION['__flash_data'][$key])) {
                unset($_SESSION['__flash_data'][$key]);
            }
            return $value;
        };
        $this->session->userdata = function($key = null) {
            if ($key === null) {
                return $_SESSION;
            }
            return isset($_SESSION[$key]) ? $_SESSION[$key] : null;
        };
        $this->session->sess_destroy = function() {
            if (isset($_SESSION)) {
                $_SESSION = array();
                session_destroy();
            }
        };
    }

    private function try_initialize_database()
    {
        // Suppress all database-related errors and output
        $original_error_reporting = error_reporting(0);
        $original_display_errors = ini_get('display_errors');
        ini_set('display_errors', 0);
        
        try {
            // Capture any output from database loading
            ob_start();
            
            // Try to load database
            $this->load->database();
            
            // Test connection with a simple query
            $result = @$this->db->query("SELECT 1 as test");
            
            if ($result && $this->db->conn_id) {
                $this->db_connected = true;
                $this->use_database = true;
                
                // Try to load model
                try {
                    $this->load->model('m_data');
                } catch (Exception $e) {
                    // Model loading failed, but database works
                }
            } else {
                throw new Exception("Database connection test failed");
            }
            
            // Clean any output that was generated during successful connection
            ob_end_clean();
            
        } catch (Exception $e) {
            // Clean any output that was generated during failed connection
            if (ob_get_level()) {
                ob_end_clean();
            }
            
            $this->db_connected = false;
            $this->use_database = false;
            
            // Unset database object to prevent further errors
            if (isset($this->db)) {
                unset($this->db);
            }
        }
        
        // Restore original error reporting settings
        error_reporting($original_error_reporting);
        ini_set('display_errors', $original_display_errors);
    }

    public function index()
    {
        // Clean any existing output
        if (ob_get_level()) {
            ob_clean();
        }

        // Set validation rules
        $this->form_validation->set_rules('username', 'Username', 'required|trim', ['required' => 'Username harus di isi!']);
        $this->form_validation->set_rules('password', 'Password', 'required|trim', ['required' => 'Password harus di isi!']);

        if ($this->form_validation->run() == TRUE) {
            // Get and sanitize input data
            $username = htmlspecialchars($this->input->post('username', TRUE), ENT_QUOTES, 'UTF-8');
            $password = htmlspecialchars($this->input->post('password', TRUE), ENT_QUOTES, 'UTF-8');

            // Authenticate user
            if ($this->use_database) {
                $this->authenticate_with_database($username, $password);
            } else {
                $this->process_successful_login($username, $password);
            }
        } else {
            // Load login view
            $this->load_login_view();
        }
    }

private function authenticate_with_database($username, $password)
{
    try {
        // Debug: Cek koneksi database
        if (!$this->db_connected || !isset($this->db)) {
            throw new Exception("Database tidak terhubung");
        }

        $login_success = false;
        $user_data = null;
        $user_type = '';

        // Cek login admin
        $this->db->where('username', $username);
        $admin = $this->db->get('tb_admin')->row_array();
        
        if ($admin) {
            // Debug log untuk melihat data yang ditemukan
            log_message('debug', 'Admin found: ' . json_encode($admin));
            
            if ($this->verify_password($password, $admin['password'])) {
                $user_data = $admin;
                $user_type = 'admin';
                $login_success = true;
            }
        }

        // Jika admin tidak berhasil, cek guru
        if (!$login_success) {
            $this->db->where('username', $username);
            $guru = $this->db->get('tb_guru')->row_array();
            
            if ($guru) {
                log_message('debug', 'Guru found: ' . json_encode($guru));
                
                if ($this->verify_password($password, $guru['password'])) {
                    $user_data = $guru;
                    $user_type = 'guru';
                    $login_success = true;
                }
            }
        }

        // Jika guru tidak berhasil, cek siswa
        if (!$login_success) {
            $this->db->where('username', $username);
            $siswa = $this->db->get('tb_siswa')->row_array();
            
            if ($siswa) {
                log_message('debug', 'Siswa found: ' . json_encode($siswa));
                
                if ($this->verify_password($password, $siswa['password'])) {
                    $user_data = $siswa;
                    $user_type = 'siswa';
                    $login_success = true;
                }
            }
        }

        // Proses login jika berhasil
        if ($login_success && $user_data) {
            $this->process_successful_login($user_data, $user_type);
        } else {
            // Login gagal - Debug info
            log_message('debug', 'Login failed for username: ' . $username);
            $this->set_flash_message('<div class="alert alert-danger alert-message text-center"><b>Login Gagal !,<br></b> Username atau Password tidak ditemukan</div>');
            $this->safe_redirect('auth');
        }
        
    } catch (Exception $e) {
        log_message('error', 'Database authentication error: ' . $e->getMessage());
        $this->set_flash_message('<div class="alert alert-danger alert-message text-center"><b>Terjadi kesalahan sistem:</b> ' . $e->getMessage() . '</div>');
        $this->safe_redirect('auth');
    }
}

private function process_successful_login($user_data, $user_type)
{
    switch ($user_type) {
        case 'admin':
            $data_session = array(
                'id'        => $user_data['id'],
                'nama'      => $user_data['nama_user'],
                'username'  => $user_data['username'],
                'status'    => 'admin_login'
            );
            $redirect_url = 'home';
            break;
            
        case 'guru':
        $data_session = array(
            'id'        => $user_data['id'],
            'id_guru'   => $user_data['id_guru'],
            'nama'      => $user_data['nama_guru'],
            'username'  => $user_data['username'],
            'status'    => 'guru_login'
        );
        $redirect_url = 'home';
        break;
            
        case 'siswa':
            $data_session = array(
                'id'        => $user_data['id_siswa'],
                'nis'       => $user_data['nis'],
                'nama'      => $user_data['nama_siswa'],
                'username'  => $user_data['username'],
                'status'    => 'siswa_login'
            );
            $redirect_url = 'home_siswa';
            break;
            
        default:
            throw new Exception("User type tidak valid");
    }
    
    $this->set_session_data($data_session);
    $this->set_flash_message('<div class="alert alert-success alert-message text-center"><b>Login Berhasil !,<br></b> Halaman ini akan dialihkan ke Halaman Home</div>');
    redirect(base_url($redirect_url));
}

    private function verify_password($input_password, $stored_password)
{
    // Jika password di database dalam bentuk hash
    if (function_exists('password_verify') && (
        substr($stored_password, 0, 4) === '$2y$' || 
        substr($stored_password, 0, 4) === '$2a$' ||
        substr($stored_password, 0, 4) === '$2b$'
    )) {
        return password_verify($input_password, $stored_password);
    }
    
    // Plain text comparison (untuk sistem lama)
    return $input_password === $stored_password;
}

// Method tambahan untuk debugging
private function debug_database_connection()
{
    if (!$this->db_connected) {
        return "Database tidak terhubung";
    }
    
    try {
        // Test query
        $query = $this->db->query("SELECT COUNT(*) as total FROM tb_admin");
        $result = $query->row();
        return "Database OK - Admin count: " . $result->total;
    } catch (Exception $e) {
        return "Database error: " . $e->getMessage();
    }
}

    private function set_session_data($data)
    {
        if (is_object($this->session)) {
            if (method_exists($this->session, 'set_userdata')) {
                $this->session->set_userdata($data);
            } else {
                // Use closure/function
                call_user_func($this->session->set_userdata, $data);
            }
        } else {
            // Fallback to native PHP session
            foreach ($data as $key => $value) {
                $_SESSION[$key] = $value;
            }
        }
    }

    private function set_flash_message($message)
    {
        if (is_object($this->session)) {
            if (method_exists($this->session, 'set_flashdata')) {
                $this->session->set_flashdata('message', $message);
            } else {
                // Use closure/function
                call_user_func($this->session->set_flashdata, 'message', $message);
            }
        } else {
            // Fallback to native PHP session
            if (!isset($_SESSION['__flash_data'])) {
                $_SESSION['__flash_data'] = [];
            }
            $_SESSION['__flash_data']['message'] = $message;
        }
    }

    private function safe_redirect($url)
    {
        // Clean all output buffers
        while (ob_get_level()) {
            ob_end_clean();
        }
        
        // Use base_url if available, otherwise construct manually
        if (function_exists('base_url')) {
            redirect(base_url($url));
        } else {
            // Manual redirect construction
            $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
            $base_url = $protocol . $_SERVER['HTTP_HOST'] . dirname($_SERVER['SCRIPT_NAME']) . '/';
            header('Location: ' . $base_url . $url);
            exit();
        }
    }

    private function load_login_view()
    {
        $data['title'] = 'Login';
        
        try {
            if (file_exists(APPPATH . 'views/v_login.php')) {
                $this->load->view('v_login', $data);
            } else {
                $this->show_default_login_form();
            }
        } catch (Exception $e) {
            $this->show_default_login_form();
        }
    }

    private function show_default_login_form()
    {
        // Clean any existing output
        if (ob_get_level()) {
            ob_clean();
        }
        
        // Get flash message from session
        $flash_message = '';
        if (isset($_SESSION['__flash_data']['message'])) {
            $flash_message = $_SESSION['__flash_data']['message'];
            unset($_SESSION['__flash_data']['message']);
        }
        
        // Get validation errors
        $errors = '';
        if (function_exists('validation_errors')) {
            $errors = validation_errors();
        }
        
        $html = '<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login Sistem</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body { 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); 
            min-height: 100vh; 
            font-family: "Segoe UI", Tahoma, Geneva, Verdana, sans-serif;
        }
        .login-container { 
            max-width: 420px; 
            margin: 0 auto; 
            padding-top: 80px; 
        }
        .card { 
            border: none; 
            border-radius: 20px; 
            box-shadow: 0 15px 35px rgba(0,0,0,0.1), 0 5px 15px rgba(0,0,0,0.07);
            backdrop-filter: blur(10px);
            background: rgba(255, 255, 255, 0.9);
        }
        .card-header { 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); 
            color: white; 
            border-radius: 20px 20px 0 0 !important;
            padding: 1.5rem;
        }
        .btn-primary { 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); 
            border: none;
            border-radius: 10px;
            padding: 12px;
            font-weight: 600;
            transition: all 0.3s ease;
        }
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }
        .form-control {
            border-radius: 10px;
            border: 2px solid #e1e5e9;
            padding: 12px 15px;
            transition: all 0.3s ease;
        }
        .form-control:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
        }
        .demo-info { 
            font-size: 0.85em; 
            color: #666; 
            margin-top: 20px;
            background: #f8f9fa;
            padding: 15px;
            border-radius: 10px;
        }
        .connection-status { 
            margin-bottom: 20px; 
            padding: 12px; 
            border-radius: 10px;
            font-weight: 500;
        }
        .connected { 
            background-color: #d4edda; 
            color: #155724; 
            border: 1px solid #c3e6cb; 
        }
        .disconnected { 
            background-color: #f8d7da; 
            color: #721c24; 
            border: 1px solid #f5c6cb; 
        }
        .alert {
            border-radius: 10px;
            border: none;
        }
        .logo {
            font-size: 2rem;
            margin-bottom: 0.5rem;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="login-container">
            <div class="card">
                <div class="card-header text-center">
                    <div class="logo">🔐</div>
                    <h4 class="mb-0">Login Sistem</h4>
                    <small class="opacity-75">Selamat datang kembali</small>
                </div>
                <div class="card-body p-4">
                    <div class="connection-status ' . ($this->use_database ? 'connected' : 'disconnected') . '">
                        <strong>Status Database:</strong> ' . ($this->use_database ? '✓ Terhubung' : '✗ Mode Demo') . '
                    </div>
                    
                    ' . $flash_message . '
                    ' . ($errors ? '<div class="alert alert-danger">' . $errors . '</div>' : '') . '
                    
                    <form method="post" action="' . $_SERVER['REQUEST_URI'] . '">
                        <div class="mb-3">
                            <label for="username" class="form-label fw-bold">Username</label>
                            <input type="text" class="form-control" id="username" name="username" required autocomplete="username">
                        </div>
                        <div class="mb-3">
                            <label for="password" class="form-label fw-bold">Password</label>
                            <input type="password" class="form-control" id="password" name="password" required autocomplete="current-password">
                        </div>
                        <button type="submit" class="btn btn-primary w-100">
                            <i class="fas fa-sign-in-alt me-2"></i> Masuk
                        </button>
                    </form>
                    
                    ' . (!$this->use_database ? '
                    <div class="demo-info text-center">
                        <hr class="my-3">
                        <strong>🚀 Mode Demo</strong><br>
                        <small class="text-muted mb-2 d-block">Database tidak tersambung, gunakan akun demo:</small>
                        <div class="row g-2 text-start">
                            <div class="col-12"><strong>Admin:</strong> <code>admin</code> / <code>admin123</code></div>
                            <div class="col-12"><strong>Guru:</strong> <code>guru</code> / <code>guru123</code></div>
                            <div class="col-12"><strong>Siswa:</strong> <code>siswa</code> / <code>siswa123</code></div>
                        </div>
                    </div>
                    ' : '') . '
                </div>
            </div>
            
            <div class="text-center mt-3">
                <small class="text-white opacity-75">
                    ' . ($this->use_database ? 'Sistem Online' : 'Mode Demonstrasi') . ' © ' . date('Y') . '
                </small>
            </div>
        </div>
    </div>
</body>
</html>';

        echo $html;
        
        // End output buffer
        if (ob_get_level()) {
            ob_end_flush();
        }
    }

    public function logout()
    {
        // Clean output buffer
        if (ob_get_level()) {
            ob_clean();
        }

        // Destroy session
        if (is_object($this->session)) {
            if (method_exists($this->session, 'sess_destroy')) {
                $this->session->sess_destroy();
            } else {
                call_user_func($this->session->sess_destroy);
            }
        } else {
            if (isset($_SESSION)) {
                $_SESSION = array();
                session_destroy();
            }
        }
        
        $this->safe_redirect('auth');
    }
}