<?php
defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Base Controller untuk menangani error database dan session
 * Letakkan file ini di application/core/MY_Controller.php
 */
class MY_Controller extends CI_Controller
{
    protected $db_connected = false;
    protected $use_database = false;
    protected $session_loaded = false;

    public function __construct()
{
    // Suppress all warnings and errors during initialization
    $original_error_reporting = error_reporting(E_ERROR | E_PARSE);
    $original_display_errors = ini_get('display_errors');
    ini_set('display_errors', 0);
    
    // Start output buffering to catch any unwanted output
    if (!ob_get_level()) {
        ob_start();
    }

    parent::__construct();
    
    // Load essential libraries and helpers
    $this->load->helper('url');
    
    // Load form validation with error handling
    try {
        $this->load->library('form_validation');
    } catch (Exception $e) {
        // Continue without form validation if it fails
    }
    
    $this->load->helper('security');
    
    // Initialize session handling
    $this->initialize_session();
    
    // Try to load database and model (silently handle errors)
    $this->try_initialize_database();
    
    // Restore original error reporting
    error_reporting($original_error_reporting);
    ini_set('display_errors', $original_display_errors);
}

    protected function initialize_session()
    {
        // Load CI session library with error handling
        try {
            // Suppress output during session loading
            ob_start();
            $this->load->library('session');
            $this->session_loaded = true;
            ob_end_clean();
        } catch (Exception $e) {
            // Clean buffer if there was output
            if (ob_get_level()) {
                ob_end_clean();
            }
            
            // If CI session fails, start native PHP session as fallback
            if (session_status() == PHP_SESSION_NONE) {
                ini_set('session.cookie_httponly', 1);
                ini_set('session.use_strict_mode', 1);
                ini_set('session.cookie_secure', 0);
                @session_start();
            }
            
            // Create mock session object for compatibility
            $this->create_mock_session();
            $this->session_loaded = false;
        }
    }

    protected function create_mock_session()
    {
        $this->session = new stdClass();
        $this->session->set_userdata = function($data) {
            if (is_array($data)) {
                foreach ($data as $key => $value) {
                    $_SESSION[$key] = $value;
                }
            }
        };
        $this->session->set_flashdata = function($key, $value) {
            if (!isset($_SESSION['__flash_data'])) {
                $_SESSION['__flash_data'] = [];
            }
            $_SESSION['__flash_data'][$key] = $value;
        };
        $this->session->flashdata = function($key) {
            $value = isset($_SESSION['__flash_data'][$key]) ? $_SESSION['__flash_data'][$key] : null;
            if (isset($_SESSION['__flash_data'][$key])) {
                unset($_SESSION['__flash_data'][$key]);
            }
            return $value;
        };
        $this->session->userdata = function($key = null) {
            if ($key === null) {
                return $_SESSION;
            }
            return isset($_SESSION[$key]) ? $_SESSION[$key] : null;
        };
        $this->session->sess_destroy = function() {
            if (isset($_SESSION)) {
                $_SESSION = array();
                session_destroy();
            }
        };
    }

    protected function try_initialize_database()
{
    // Suppress all database-related errors and output
    $original_error_reporting = error_reporting(0);
    $original_display_errors = ini_get('display_errors');
    ini_set('display_errors', 0);
    
    // Set custom error handler to suppress database warnings
    set_error_handler(function($errno, $errstr, $errfile, $errline) {
        // Suppress all database related errors
        return true;
    });
    
    try {
        // Capture any output from database loading
        ob_start();
        
        // Try to load database with complete error suppression
        @$this->load->database();
        
        // Test connection with a simple query
        if (isset($this->db) && @$this->db->conn_id) {
            $result = @$this->db->query("SELECT 1 as test");
            if ($result) {
                $this->db_connected = true;
                $this->use_database = true;
                
                // Try to load model
                try {
                    @$this->load->model('m_data');
                } catch (Exception $e) {
                    // Model loading failed, but database works
                }
            } else {
                throw new Exception("Database query test failed");
            }
        } else {
            throw new Exception("Database connection failed");
        }
        
        // Clean any output that was generated during successful connection
        ob_end_clean();
        
    } catch (Exception $e) {
        // Clean any output that was generated during failed connection
        if (ob_get_level()) {
            ob_end_clean();
        }
        
        $this->db_connected = false;
        $this->use_database = false;
        
        // Unset database object to prevent further errors
        if (isset($this->db)) {
            unset($this->db);
        }
        
        // Log the error if logging is available
        if (function_exists('log_message')) {
            @log_message('info', 'Database connection failed, running in demo mode');
        }
    }
    
    // Restore error handler and settings
    restore_error_handler();
    error_reporting($original_error_reporting);
    ini_set('display_errors', $original_display_errors);
}

    /**
     * Safe method to get session data
     */
    protected function get_session_data($key = null)
    {
        try {
            if (is_object($this->session)) {
                if (method_exists($this->session, 'userdata')) {
                    return $this->session->userdata($key);
                } else {
                    // Use closure/function
                    return call_user_func($this->session->userdata, $key);
                }
            } else {
                // Fallback to native PHP session
                if ($key === null) {
                    return $_SESSION ?? [];
                }
                return $_SESSION[$key] ?? null;
            }
        } catch (Exception $e) {
            return null;
        }
    }

    /**
     * Safe method to set session data
     */
    protected function set_session_data($data)
    {
        if (is_object($this->session)) {
            if (method_exists($this->session, 'set_userdata')) {
                $this->session->set_userdata($data);
            } else {
                // Use closure/function
                call_user_func($this->session->set_userdata, $data);
            }
        } else {
            // Fallback to native PHP session
            foreach ($data as $key => $value) {
                $_SESSION[$key] = $value;
            }
        }
    }

    /**
     * Check if user is logged in
     */
    protected function is_logged_in()
    {
        $status = $this->get_session_data('status');
        return in_array($status, ['admin_login', 'guru_login', 'siswa_login']);
    }

    /**
     * Get current user type
     */
    protected function get_user_type()
    {
        $status = $this->get_session_data('status');
        switch ($status) {
            case 'admin_login':
                return 'admin';
            case 'guru_login':
                return 'guru';
            case 'siswa_login':
                return 'siswa';
            default:
                return null;
        }
    }

    /**
     * Redirect if not logged in
     */
    protected function require_login()
    {
        if (!$this->is_logged_in()) {
            $this->safe_redirect('auth');
        }
    }

    /**
     * Safe redirect method
     */
    protected function safe_redirect($url)
    {
        // Clean all output buffers
        while (ob_get_level()) {
            ob_end_clean();
        }
        
        // Use base_url if available, otherwise construct manually
        if (function_exists('base_url')) {
            redirect(base_url($url));
        } else {
            // Manual redirect construction
            $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
            $base_url = $protocol . $_SERVER['HTTP_HOST'] . dirname($_SERVER['SCRIPT_NAME']) . '/';
            header('Location: ' . $base_url . $url);
            exit();
        }
    }
    
    /**
     * Safe method to load views with fallback
     */
    protected function safe_load_view($view, $data = [], $return = FALSE)
    {
        try {
            return $this->load->view($view, $data, $return);
        } catch (Exception $e) {
            // If view loading fails, return error message
            if ($return) {
                return '<div class="alert alert-warning">View "' . $view . '" tidak ditemukan.</div>';
            } else {
                echo '<div class="alert alert-warning">View "' . $view . '" tidak ditemukan.</div>';
            }
        }
    }
}